﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using System.ServiceModel;
using VA.PPMS.Context;

namespace VA.PPMS.CRM.Plugins
{
    public class ProviderServiceUpdate : IPlugin
    {
        private ITracingService tracingService;
        private const string PluginName = "ProviderServiceUpdate";
        private const string MessageCreate = "CREATE";
        private const string MessageUpdate = "UPDATE";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (entity.LogicalName != "ppms_providerservice")
                    return;

                tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {

                    var source = new Entity();

                    // handle event based on message type
                    switch (context.MessageName.ToUpper())
                    {
                        case MessageCreate:
                            tracingService.Trace("Create: update corresponding provider property");
                            source = GetSourceEntity(service, entity.Id);
                            SetProviderProperty(service, source, true);
                            //Moved to separate plugin
                            //UpdateProviderServiceFields(service, entity);
                            break;

                        case MessageUpdate:
                            tracingService.Trace("Check for existing services, and set property");
                            if (entity.Contains("statecode"))
                            {
                                source = GetSourceEntity(service, entity.Id);
                                UpdateProviderProperty(service, source);
                            }
                            //Moved to separate plugin
                            /*
                            if (entity.Contains("ppms_updateprovservicefields") || entity.Contains("ppms_providerid") || entity.Contains("ppms_specialty") || entity.Contains("ppms_caresite"))
                            {
                                //UpdateProviderServiceFields(service, entity);
                            }
                            */
                            break;

                        default:
                            tracingService.Trace("Valid event message not found");
                            break;
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            tracingService.Trace("Done");
        }

        private Entity GetSourceEntity(IOrganizationService service, Guid entityId)
        {
            var request = new RetrieveRequest();

            request.Target = new EntityReference("ppms_providerservice", entityId);
            request.ColumnSet = new ColumnSet(new string[] { "ppms_providerserviceid", "ppms_name", "ppms_network", "ppms_providerid", "statuscode" });
            request.RelatedEntitiesQuery = new RelationshipQueryCollection();

            // Retrieve related entities
            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_ppms_providerservice"),
                new QueryExpression("account")
                {
                    ColumnSet = new ColumnSet("accountid", "name", "ppms_affiliationtype_triwest")
                }
            );

            request.RelatedEntitiesQuery.Add(new Relationship("ppms_vaprovidernetwork_providerservice_network"),
                new QueryExpression("ppms_vaprovidernetwork")
                {
                    ColumnSet = new ColumnSet("ppms_vaprovidernetworkid", "ppms_name")
                }
            );

            //Get response
            var response = (RetrieveResponse)service.Execute(request);
            if (response != null)
                return response.Entity;

            return null;
        }

        private Entity GetProvider(IOrganizationService service, Entity source, string targetProperty)
        {
            var entity = source.GetAttributeValue<EntityReference>("ppms_providerid");
            if (entity != null)
            {
                return service.Retrieve("account", entity.Id, new ColumnSet(new string[] { "accountid", "name", targetProperty }));
            }

            return null;
        }

        private string GetNetworkProperty(IOrganizationService service, Entity source)
        {
            string targetProperty = String.Empty;

            // determine target network
            var children = source.RelatedEntities[new Relationship("ppms_vaprovidernetwork_providerservice_network")];
            if (children != null && children.Entities != null)
            {
                foreach (var child in children.Entities)
                {
                    targetProperty = child.GetAttributeValue<string>("ppms_name");
                    targetProperty = targetProperty.Replace(" ", "").ToLower();
                }

                if (!String.IsNullOrEmpty(targetProperty))
                {
                    targetProperty = String.Format("ppms_affiliationtype_{0}", targetProperty);
                }
            }

            return targetProperty;
        }

        private EntityCollection GetNetworks(IOrganizationService service)
        {
            FilterExpression filter = new FilterExpression();
            filter.AddCondition("statecode", ConditionOperator.Equal, (int)PpmsHelper.AccountState.Active);

            QueryExpression query = new QueryExpression("ppms_vaprovidernetwork");
            query.ColumnSet.AddColumns("ppms_name");
            query.Criteria.AddFilter(filter);

            return service.RetrieveMultiple(query);
        }

        private EntityCollection GetActiveServicesByProvider(IOrganizationService service, Guid providerId, Guid networkId)
        {
            FilterExpression filter = new FilterExpression();
            filter.AddCondition("statecode", ConditionOperator.Equal, (int)PpmsHelper.AccountState.Active);
            filter.AddCondition("ppms_providerid", ConditionOperator.Equal, providerId);
            filter.AddCondition("ppms_network", ConditionOperator.Equal, networkId);

            QueryExpression query = new QueryExpression("ppms_providerservice");
            query.ColumnSet.AddColumns("ppms_name");
            query.Criteria.AddFilter(filter);

            return service.RetrieveMultiple(query);
        }

        private void SetProviderProperty(IOrganizationService service, Entity entity, bool newStatus)
        {
            tracingService.Trace("Create: Get network");
            string property = GetNetworkProperty(service, entity);
            tracingService.Trace("Create: Get provider");
            var provider = GetProvider(service, entity, property);
            tracingService.Trace("Create: Get status");
            bool? status = provider.GetAttributeValue<bool?>(property);

            // check if change needs to be made
            if (!status.HasValue || status.Value != newStatus)
            {
                // set target attribute based on new service
                if (provider.Attributes.Contains(property))
                {
                    tracingService.Trace("Create: contains {0}", property);
                    provider.Attributes[property] = newStatus;
                }
                else
                {
                    tracingService.Trace("Create: does not contain {0}", property);
                    provider.Attributes.Add(property, newStatus);
                }
                tracingService.Trace("Create: Update");
                // save changes
                service.Update(provider);
            }
        }

        private void UpdateProviderProperty(IOrganizationService service, Entity entity)
        {
            var provider = entity.GetAttributeValue<EntityReference>("ppms_providerid");
            var network = entity.GetAttributeValue<EntityReference>("ppms_network");

            // Determine new state based on the state of the record
            var list = GetActiveServicesByProvider(service, provider.Id, network.Id);
            // If active records still exist, checkbox is on
            var newState = list != null && list.Entities != null && list.Entities.Any();
            SetProviderProperty(service, entity, newState);
        }


        private void UpdateProviderServiceFields(IOrganizationService service, Entity entity)
        {           
            using (var svc = new PpmsContext(service))
            {
                var updateProvService = new ppms_providerservice
                {
                    Id = entity.Id
                };

                // Get Provider Service and needed attributes      
                var provService = service.Retrieve("ppms_providerservice", entity.Id, new ColumnSet(new string[] { "ppms_providerid", "ppms_specialty", "ppms_caresite"}));
                tracingService.Trace("Provider Service retrieved: {0}", provService.Id.ToString());

                //Get Care Site Entity Ref. 
                var careSiteReference = provService.GetAttributeValue<EntityReference>("ppms_caresite");

                if (careSiteReference != null)
                {
                    tracingService.Trace("Care Site Reference found: {0}", careSiteReference.Id.ToString());
                    var careSite = service.Retrieve("ppms_caresite", careSiteReference.Id,
                        new ColumnSet(new string[]
                        {
                            "ppms_address_line1", "ppms_address_city", "ppms_statename", "ppms_address_postalcode",
                            "ppms_address_latitude", "ppms_address_longitude"
                        }));
                    tracingService.Trace("Care Site found: {0}", careSite.Id.ToString());

                    //Look up values and assign to the Provider Service
                    var careSiteAddress = careSite.GetAttributeValue<string>("ppms_address_line1");
                    if (careSiteAddress != null)
                    {
                        updateProvService.ppms_caresiteaddress = careSiteAddress;
                        tracingService.Trace("Care Site Address Found: {0}", careSiteAddress);
                    }

                    var careSiteCity = careSite.GetAttributeValue<string>("ppms_address_city");
                    if (careSiteCity != null)
                    {
                        updateProvService.ppms_caresitecity = careSiteCity;
                        tracingService.Trace("Care Site City Found: {0}", careSiteCity);
                    }

                    var careSiteState = careSite.GetAttributeValue<string>("ppms_statename");
                    if (careSiteState != null)
                    {
                        updateProvService.ppms_caresitestateprovince = careSiteState;
                        tracingService.Trace("Care Site State/Province Found: {0}", careSiteState);
                    }

                    var careSitePostalCode = careSite.GetAttributeValue<string>("ppms_address_postalcode");
                    if (careSitePostalCode != null)
                    {
                        updateProvService.ppms_caresitezipcode = careSitePostalCode;
                        tracingService.Trace("Care Site Zip Found: {0}", careSitePostalCode);
                    }

                    var careSiteLatitude = careSite.GetAttributeValue<string>("ppms_address_latitude");
                    if (careSiteLatitude != null)
                    {
                        updateProvService.ppms_CareSiteAddressLatitude = careSiteLatitude;
                        tracingService.Trace("Care Site Address Found: {0}", careSiteLatitude);
                    }

                    var careSiteLongitude = careSite.GetAttributeValue<string>("ppms_address_longitude");
                    if (careSiteLongitude != null)
                    {
                        updateProvService.ppms_CareSiteAddressLongitude = careSiteLongitude;
                        tracingService.Trace("Care Site Address Found: {0}", careSiteLongitude);
                    }
                }
                else
                {
                    tracingService.Trace("Care Site Reference not found");
                }

                //Get the Provider Reference. 
                var providerReference = provService.GetAttributeValue<EntityReference>("ppms_providerid");
                if (providerReference != null)
                {
                    tracingService.Trace("Provider Reference found: {0}", providerReference.Id.ToString());
                    //Retrieve the Provider and needed attributes           
                    var provider = service.Retrieve("account", providerReference.Id,
                        new ColumnSet(new string[]
                        {
                            "name", "ppms_gender", "ppms_individualisacceptingnewpatients",
                            "ppms_primarycareprovideracceptingva", "ppms_primarycarephysician"
                        }));
                    tracingService.Trace("Provider found: {0}", provider.Id.ToString());

                    var providerName = provider.GetAttributeValue<string>("name");
                    if (providerName != null)
                    {
                        updateProvService.ppms_providername = providerName;
                        tracingService.Trace("Provider Name Found: {0}", providerName);
                    }

                    var providerGender = provider.GetAttributeValue<OptionSetValue>("ppms_gender");
                    if (providerGender != null)
                    {
                        updateProvService.ppms_providergender = providerGender;
                        tracingService.Trace("Provider Gender retrieved: {0}",
                            providerGender.Value.ToString());
                    }

                    var providerIndAcceptingNewPatients = provider.GetAttributeValue<bool>("ppms_individualisacceptingnewpatients");
                    updateProvService.ppms_provideracceptingnewpatients = providerIndAcceptingNewPatients;
                    tracingService.Trace("Provider Accepting New Patients Value retrieved: {0}",
                        providerIndAcceptingNewPatients.ToString());

                    var providerPrimaryCareAcceptingVa = provider.GetAttributeValue<bool>("ppms_primarycareprovideracceptingva");
                    updateProvService.ppms_provideracceptingva = providerPrimaryCareAcceptingVa;
                    tracingService.Trace("Provider Primary Care & Accepting VA retrieved: {0}",
                        providerPrimaryCareAcceptingVa.ToString());

                    var providerPrimaryCare = provider.GetAttributeValue<bool>("ppms_primarycarephysician");
                    updateProvService.ppms_providerisprimarycare = providerPrimaryCare;
                    tracingService.Trace("Provider Primary Care Value retrieved: {0}",
                        providerPrimaryCare.ToString());
                }
                else
                {
                    tracingService.Trace("Provider Reference not found");
                }

                //Get Specialty Entity Ref. 
                var specialtyReference = provService.GetAttributeValue<EntityReference>("ppms_specialty");
                if (specialtyReference != null)
                {
                    tracingService.Trace("Specialty Reference found: {0}", specialtyReference.Id.ToString());
                    //Retrieve the Specialty and needed attributes
                    var specialty = service.Retrieve("ppms_taxonomy", specialtyReference.Id,
                        new ColumnSet(new string[] {"ppms_specialtycode"}));
                    tracingService.Trace("Specialty found: {0}", specialty.Id.ToString());

                    var specialtyCode = specialty.GetAttributeValue<string>("ppms_specialtycode");
                    if (specialtyCode != null)
                    {
                        updateProvService.ppms_specialtycode = specialtyCode;
                        tracingService.Trace("Specialty Code Found: {0}", specialtyCode);
                    }
                }
                else
                {
                    tracingService.Trace("Specialty Reference not found");
                }

                //Update the Provider Service record with new values. 
                service.Update(updateProvService);
                tracingService.Trace("Provider Service Fields Updated");
            }
        }
    }
}
